require "datafilter-test"
local Filter = require "datafilter"
local testcase = TestCase("Algorithm adler32")

local misc_mapping, progressive_adler32_expected

function testcase:test_trivial_obj ()
    local obj = Filter:new("adler32")
    is("00000001", bytes_to_hex(obj:result()))
    local obj = Filter:new("adler32")
    obj:add("")
    is("00000001", bytes_to_hex(obj:result()))
end

function testcase:test_misc_adler32 ()
    for input, expected in pairs(misc_mapping) do
        local got = Filter.adler32(input)
        is(4, got:len())
        is(expected, bytes_to_hex(got),
           "Adler32 of " .. string.format("%q", input))
    end
end

function testcase:test_gradual_size_increase ()
    local input = ""
    local byte = 7

    for i = 1, #progressive_adler32_expected do
        local expected = progressive_adler32_expected[i]

        input = input .. string.char(byte)
        byte = (byte + 23) % 256
        assert(input:len() == i)

        local got = Filter.adler32(input)
        is(4, got:len())
        is(expected, bytes_to_hex(got),
           "Adler32 of " .. string.format("%q", input))
    end
end

misc_mapping = {
    -- Test data from the test suite of the Digest::Adler32 Perl module
    [""] = "00000001",
    ["a"] = "00620062",
    ["abc"] = "024d0127",
    ["abcabc"] = "080c024d",
    [("\255"):rep(32)] = "0e2e1fe1",
}

-- Data calculated with a bit of Perl code (in the file 't/adler32-gen.pl')
-- where each entry in the array is for a chunk of data of as many bytes as its
-- index, and the bytes all progress in a simple way.
progressive_adler32_expected = {
    "00080008",
    "002e0026",
    "0089005b",
    "013000a7",
    "023a010a",
    "03be0184",
    "05d30215",
    "089002bd",
    "0c0c037c",
    "105e0452",
    "159d053f",
    "1ae00543",
    "203e055e",
    "25ce0590",
    "2ba705d9",
    "31e00639",
    "389006b0",
    "3fce073e",
    "47b107e3",
    "5050089f",
    "59c20972",
    "641e0a5c",
    "6e7b0a5d",
    "78f00a75",
    "83940aa4",
    "8e7e0aea",
    "99c50b47",
    "a5800bbb",
    "b1c60c46",
    "beae0ce8",
    "cc4f0da1",
    "dac00e71",
    "ea180f58",
    "fa6e1056",
    "0ae8106b",
    "1b7f1097",
    "2c5910da",
    "3d8d1134",
    "4f3211a5",
    "615f122d",
    "742b12cc",
    "87ad1382",
    "9bfc144f",
    "b12f1533",
    "c75d162e",
    "dd9d1640",
    "f4061669",
    "0abe16a9",
    "21be1700",
    "392c176e",
    "511f17f3",
    "69ae188f",
    "82f01942",
    "9cfc1a0c",
    "b7e91aed",
    "d3ce1be5",
    "efc21bf4",
    "0beb1c1a",
    "28421c57",
    "44ed1cab",
    "62031d16",
    "7f9b1d98",
    "9dcc1e31",
    "bcad1ee1",
    "dc551fa8",
    "fcdb2086",
    "1e65217b",
    "3fec2187",
    "619621aa",
    "837a21e4",
    "a5af2235",
    "c84c229d",
    "eb68231c",
    "0f2923b2",
    "3388245f",
    "58ab2523",
    "7ea925fe",
    "a59926f0",
    "cc9226f9",
    "f3ab2719",
    "1b0a2750",
    "42a8279e",
    "6aab2803",
    "932a287f",
    "bc3c2912",
    "e5f829bc",
    "10842a7d",
    "3bd92b55",
    "681d2c44",
    "94672c4a",
    "c0ce2c67",
    "ed692c9b",
    "1a5e2ce6",
    "47a62d48",
    "75672dc1",
    "a3b82e51",
    "d2b02ef8",
    "02752fb6",
    "3300308b",
    "64773177",
    "95f1317a",
    "c7853194",
    "f94a31c5",
    "2b66320d",
    "5dd2326c",
    "90b432e2",
    "c423336f",
    "f8363413",
    "2d1334ce",
    "62b335a0",
    "993c3689",
    "cfc53689",
    "067436a0",
    "3d4236ce",
    "74553713",
    "abc4376f",
    "e3a637e2",
    "1c21386c",
    "552e390d",
    "8ef339c5",
    "c9873a94",
    "05103b7a",
    "41873c77",
    "7e123c8b",
    "bac83cb6",
    "f7c03cf8",
    "35203d51",
    "72e13dc1",
    "b1293e48",
    "f00f3ee6",
    "2fb93f9b",
    "70204067",
    "b16a414a",
    "f3ae4244",
    "36124255",
    "788f427d",
    "bb4b42bc",
    "fe5d4312",
    "41eb437f",
    "85ee4403",
    "ca8c449e",
    "0feb4550",
    "56044619",
    "9cfd46f9",
    "e4ed47f0",
    "2cfa47fe",
    "751d4823",
    "bd7c485f",
    "063d48b2",
    "4f59491c",
    "98f6499d",
    "e32b4a35",
    "2e1e4ae4",
    "79c84baa",
    "c64f4c87",
    "13d94d7b",
    "615f4d86",
    "af074da8",
    "fce84de1",
    "4b284e31",
    "99c04e98",
    "e8d64f16",
    "38904fab",
    "88e75057",
    "da01511a",
    "2c0451f4",
    "7ee952e5",
    "d1d652ed",
    "24f1530c",
    "78335342",
    "cbc2538f",
    "1fc453f3",
    "7432546e",
    "c9325500",
    "1eea55a9",
    "75535669",
    "cc935740",
    "24d0582e",
    "7d035833",
    "d552584f",
    "2de35882",
    "86af58cc",
    "dfdc592d",
    "399059a5",
    "93c45a34",
    "ee9e5ada",
    "4a445b97",
    "a6af5c6b",
    "04145d56",
    "616c5d58",
    "bedd5d71",
    "1c8d5da1",
    "7a755de8",
    "d8bb5e46",
    "37855ebb",
    "96cc5f47",
    "f6b65fea",
    "576960a4",
    "b8de6175",
    "1b4a625d",
    "7ea6635c",
    "e2186372",
    "45c6639f",
    "a9a963e3",
    "0df6643e",
    "72a664b0",
    "d7df6539",
    "3dc765d9",
    "a4576690",
    "0bc4675e",
    "74076843",
    "dd46693f",
    "46a76952",
    "b023697c",
    "19ef69bd",
    "84046a15",
    "ee886a84",
    "59a16b0a",
    "c5486ba7",
    "31b26c5b",
    "9ed86d26",
    "0cef6e08",
    "7bf06f01",
    "eb016f11",
    "5a486f38",
    "c9be6f76",
    "39986fcb",
    "a9cf7037",
    "1a9870ba",
    "8bec7154",
    "fdf17205",
    "70cd72cd",
    "e47973ac",
    "592a74a2",
    "cdd974af",
    "42bb74d3",
    "b7c9750e",
    "2d387560",
    "a30175c9",
    "19597649",
    "903976e0",
    "07d6778e",
    "80297853",
    "f958792f",
    "73897a22",
    "edb57a2c",
    "68117a4d",
    "e2967a85",
    "5d797ad4",
    "d8b37b3a",
    "54797bb7",
    "d0c47c4b",
    "4dc97cf6",
    "cb817db8",
    "4a217e91",
    "c9a27f81",
}

lunit.run()
-- vi:ts=4 sw=4 expandtab
