#!/usr/bin/perl

# This program generates 'algorithms.c' from 'algorithms.txt'.  You only need
# to run it after you've modified the list of algorithms.  Give the name of
# the input file on the command line.

use warnings;
use strict;

die "Usage: $0 algorithms.txt algorithms.c\n" unless @ARGV == 2;
my ($input_filename, $output_filename) = @ARGV;

my $index = 0;
my @algo;

open my $input_fh, '<', $input_filename
    or die "$0: error opening '$input_filename': $!\n";
while (<$input_fh>) {
    chomp;
    s/#.*//;
    next unless /\S/;
    my ($name, $struct, $destructor) = split ' ', $_;
    die "$input_filename:$.: bad line '$_'\n"
        unless defined $destructor;
    my $has_init_method = $struct ne '-';
    my $struct_size = $struct eq '-' ? 0 : "sizeof(${struct}State)";
    push @algo, {
        name => $name,
        struct_size => $struct_size,
        init_method => ($has_init_method ? "algo_${name}_init" : 0),
        destroy_method => ($destructor ? "algo_${name}_destroy" : 0),
        index => $index++,
    };
}

open my $out_fh, '>', $output_filename
    or die "$0: error opening output file '$output_filename': $!\n";

print $out_fh <<EOF;
/* Automatically generated by '$0' from the list of algorithms
 * in '$input_filename'.
 */

EOF

for (@algo) {
    print $out_fh "static int algowrap_$_->{name} (lua_State *L);\n";
}

print $out_fh "\nstatic const AlgorithmDefinition\n",
              "filter_algorithms[] = {\n";
for (@algo) {
    print $out_fh "    { \"$_->{name}\", algo_$_->{name},",
                  " algowrap_$_->{name},\n",
                  "      $_->{struct_size}, $_->{init_method},",
                  " $_->{destroy_method} },\n";
}
print $out_fh "};\n",
              "#define NUM_ALGO_DEFS (sizeof(filter_algorithms) /",
              " sizeof(AlgorithmDefinition))\n\n";

for (@algo) {
    print $out_fh
        "static int algowrap_$_->{name} (lua_State *L) {\n",
        "    return algo_wrapper(L, &filter_algorithms[$_->{index}]);\n",
        "}\n";
}

# vi:ts=4 sw=4 expandtab
